using System;
using System.Runtime.InteropServices;
using System.Text;

namespace Sony
{
	namespace NP
	{
		public class Dialogs
		{
			public enum EnumNpDlgResult
			{
				NP_DLG_CANCELED = 0,
				NP_DLG_OK = 1,
			};

			public enum CommerceDialogMode
			{
				CATEGORY = 0,
				PRODUCT = 1,
				PRODUCE_CODE = 2,
				CHECKOUT = 3,
				DOWNLOADLIST = 4,
				PLUS = 5,
			};

			// Structure for returning result when an NP dialog is closed.
			[StructLayout(LayoutKind.Sequential, CharSet = CharSet.Ansi, Pack = 0)]
			public struct NpDialogReturn
			{
				IntPtr _npID;						// NpID, contains data corresponding to the SceNpID structure that was selected in the dialog.
				int npIDSize;						// Size of the npID byte array.
				public bool plusAllowed;
				public EnumNpDlgResult result;		// Result, 0 = canceled, 1 = ok.

				public byte[] npID
				{
					get
					{
						byte[] bytes = new byte[npIDSize];
						Marshal.Copy(_npID, bytes, 0, npIDSize);
						return bytes;
					}
				}
			};

            [DllImport("UnityNpToolkit")]
            private static extern bool PrxNpDialogGetLastError(out ResultCode result);
            
            public static bool GetLastError(out ResultCode result)
            {
                PrxNpDialogGetLastError(out result);
                return result.lastError == ErrorCode.NP_OK;
            }

            [DllImport("UnityNpToolkit")]
			private static extern bool PrxNpIsDialogOpen();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxNpDialogFriendsList();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxNpDialogSharedPlayHistory();
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxNpDialogProfile(byte[] npID);
			[DllImport("UnityNpToolkit")]
			private static extern ErrorCode PrxNpDialogCommerce(int mode, ulong features);
			[DllImport("UnityNpToolkit")]
            private static extern ErrorCode PrxNpDialogGetResult(out NpDialogReturn result);

			public static bool IsDialogOpen
			{
				get { return PrxNpIsDialogOpen(); }
			}

            public static ErrorCode FriendsList()
			{
				return PrxNpDialogFriendsList();
			}
			
            public static ErrorCode SharedPlayHistory()
			{
				return PrxNpDialogSharedPlayHistory();
			}

			public static ErrorCode Profile(byte[] npID)
			{
				return PrxNpDialogProfile(npID);
			}

			public static ErrorCode Commerce(CommerceDialogMode mode, Requests.PlusFeature features)
			{
				return PrxNpDialogCommerce((int)mode, (ulong)features);
			}

			public static NpDialogReturn GetDialogResult()
			{
				NpDialogReturn result = new NpDialogReturn();
				PrxNpDialogGetResult(out result);
				return result;
			}

			public static event Messages.EventHandler OnDlgFriendsListClosed;
			public static event Messages.EventHandler OnDlgSharedPlayHistoryClosed;
			public static event Messages.EventHandler OnDlgProfileClosed;
			public static event Messages.EventHandler OnDlgCommerceClosed;

			public static bool ProcessMessage(NP.Messages.PluginMessage msg)
			{
				switch (msg.type)
				{
					case NP.Messages.MessageType.kNPToolKit_DlgFriendsClosed:
						if (OnDlgFriendsListClosed != null) OnDlgFriendsListClosed(msg);
						return true;
						
					case NP.Messages.MessageType.kNPToolKit_DlgSharedPlayHistoryClosed:
						if (OnDlgSharedPlayHistoryClosed != null) OnDlgSharedPlayHistoryClosed(msg);
						return true;
						
					case NP.Messages.MessageType.kNPToolKit_DlgCommerceClosed:
						if (OnDlgCommerceClosed != null) OnDlgCommerceClosed(msg);
						return true;
					
					case NP.Messages.MessageType.kNPToolKit_DlgProfileClosed:
							if (OnDlgProfileClosed != null) OnDlgProfileClosed(msg);
							break;


				}
				
				return false;
			}

		}
	}
}
